<?php
// تابع تبدیل عدد انگلیسی به فارسی
function toPersianNumbers($input) {
    $englishNumbers = ['0','1','2','3','4','5','6','7','8','9'];
    $persianNumbers = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹'];
    return str_replace($englishNumbers, $persianNumbers, $input);
}

// تابع تبدیل تاریخ میلادی به شمسی
function gregorianToJalali($gy, $gm, $gd) {
    $g_d_m = array(0,31,59,90,120,151,181,212,243,273,304,334);
    if($gy > 1600){
        $jy=979;
        $gy-=1600;
    }else{
        $jy=0;
        $gy-=621;
    }
    $gy2 = ($gm > 2)?($gy+1):$gy;
    $days = (365*$gy) + intval(($gy2+3)/4) - intval(($gy2+99)/100) + intval(($gy2+399)/400) - 80 + $gd + $g_d_m[$gm-1];
    $jy += 33*intval($days/12053);
    $days %= 12053;
    $jy += 4*intval($days/1461);
    $days %= 1461;
    if($days > 365){
        $jy += intval(($days-1)/365);
        $days = ($days-1)%365;
    }
    if($days < 186){
        $jm = 1 + intval($days/31);
        $jd = 1 + ($days % 31);
    }else{
        $jm = 7 + intval(($days-186)/30);
        $jd = 1 + (($days-186) % 30);
    }
    return [$jy, $jm, $jd];
}

// تبدیل تاریخ میلادی به شمسی
function toJalaliDate($gregorianDate){
    $parts = explode('-', $gregorianDate);
    if(count($parts) !== 3) return $gregorianDate;
    list($gy, $gm, $gd) = $parts;
    list($jy, $jm, $jd) = gregorianToJalali((int)$gy, (int)$gm, (int)$gd);
    $jalaliStr = sprintf('%04d/%02d/%02d', $jy, $jm, $jd);
    return toPersianNumbers($jalaliStr);
}

function logTaskChange($entry) {
    $logFile = __DIR__ . "/data/task_log.json";
    $logs = file_exists($logFile) ? json_decode(file_get_contents($logFile), true) : [];
    $entry['datetime'] = date("Y-m-d H:i:s");
    $entry['by'] = 'admin';
    $logs[] = $entry;
    file_put_contents($logFile, json_encode($logs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// فایل‌های داده
$dataDir = __DIR__ . "/data";
if(!file_exists($dataDir)) mkdir($dataDir);
$peopleFile = "$dataDir/people.json";
$tasksFile  = "$dataDir/tasks.json";
$assignFile = "$dataDir/assignments.json";

$people = json_decode(@file_get_contents($peopleFile), true) ?: [];
$tasks = json_decode(@file_get_contents($tasksFile), true) ?: [];
$assignments = json_decode(@file_get_contents($assignFile), true) ?: [];

// حذف وظیفه
if(isset($_GET['delete'])){
    $index = intval($_GET['delete']);
    if(isset($assignments[$index])){
        $deletedItem = $assignments[$index];
        logTaskChange([
            "action" => "delete",
            "task_id" => $deletedItem['task_id'],
            "person_id" => $deletedItem['person_id'],
            "shift" => $deletedItem['shift'] ?? '-'
        ]);

        array_splice($assignments, $index, 1);
        file_put_contents($assignFile, json_encode($assignments, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
    }
    header("Location: index.php");
    exit;
}

// ویرایش
$editIndex = -1;
$editData = ["date"=>"", "task_id"=>"", "person_id"=>"", "color"=>"black", "shift"=>""];
if(isset($_GET['edit'])){
    $editIndex = intval($_GET['edit']);
    if(isset($assignments[$editIndex])){
        $editData = $assignments[$editIndex];
        $autoShowForm = true;
    }
}

// ذخیره وظیفه
if(isset($_POST['save'])){
    $newData = [
        "date" => $_POST['date'],
        "task_id" => intval($_POST['task_id']),
        "person_id" => intval($_POST['person_id']),
        "color" => $_POST['color'],
        "shift" => $_POST['shift']
    ];

    $index = intval($_POST['index']);
    
    if($index >= 0 && isset($assignments[$index])){
        // ویرایش موجود
        $assignments[$index] = $newData;
        logTaskChange([
            "action" => "edit",
            "task_id" => $newData['task_id'],
            "person_id" => $newData['person_id'],
            "shift" => $newData['shift']
        ]);
    } else {
        // افزودن جدید
        $assignments[] = $newData;
        logTaskChange([
            "action" => "add",
            "task_id" => $newData['task_id'],
            "person_id" => $newData['person_id'],
            "shift" => $newData['shift']
        ]);
    }

    file_put_contents($assignFile, json_encode($assignments, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
    header("Location: index.php");
    exit;
}

// آماده‌سازی داده‌ها بر اساس روز هفته
$weekdays = ['شنبه', 'یکشنبه', 'دوشنبه', 'سه‌شنبه', 'چهارشنبه', 'پنجشنبه', 'جمعه'];
$tasksByWeekday = [];
foreach ($weekdays as $day) $tasksByWeekday[$day] = [];

foreach ($assignments as $a) {
    $timestamp = strtotime($a['date']);
    $dayOfWeek = date('w', $timestamp);
    $mapping = [0=>1, 1=>2, 2=>3, 3=>4, 4=>5, 5=>6, 6=>0];
    $farsiDayName = $weekdays[$mapping[$dayOfWeek]];

    $taskName = "";
    foreach($tasks as $t) if($t['task_id']==$a['task_id']) $taskName=$t['title'];

    $personName = "";
    foreach($people as $p) if($p['id']==$a['person_id']) $personName=$p['name'];

    $dateShamsi = toJalaliDate($a['date']);
    $shift = $a['shift'] ?? '-';
    $tasksByWeekday[$farsiDayName][] = "<span style='color:{$a['color']}'>".htmlspecialchars("$dateShamsi - $taskName ($personName - شیفت: $shift)")."</span>";
}

// محاسبه آمار
$totalTasks = count($assignments);
$uniquePeople = array_unique(array_column($assignments, 'person_id'));
$totalPeople = count($uniquePeople);
$uniqueTasks = array_unique(array_column($assignments, 'task_id'));
$totalTaskTypes = count($uniqueTasks);
$todayShamsi = toJalaliDate(date('Y-m-d'));

// تعداد وظایف امروز
$todayTasks = 0;
foreach($assignments as $task) {
    if($task['date'] == date('Y-m-d')) {
        $todayTasks++;
    }
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت وظایف - جدول ایام هفته</title>
    <link href="https://cdn.fontcdn.ir/Font/Persian/Vazir/Vazir.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        body {
            font-family: Vazir, sans-serif;
            background-color: #f8f9fa;
        }
        
        .marquee-container {
            width: 100%;
            overflow: hidden;
            background: #f0f8ff;
            padding: 15px 0;
            margin-bottom: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .marquee-content {
            display: inline-block;
            white-space: nowrap;
            animation: marquee 60s linear infinite;
        }
        
        .marquee-item {
            display: inline-block;
            margin: 0 40px;
            font-size: 16px;
            font-weight: 500;
            padding: 5px 10px;
            border-radius: 5px;
            background: rgba(255,255,255,0.2);
        }
        
        @keyframes marquee {
            0% { transform: translateX(0); }
            100% { transform: translateX(-50%); }
        }
        
        .info-billboard {
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            border-radius: 8px;
            padding: 15px;
        }
        
        .billboard-item {
            display: flex;
            align-items: center;
            margin: 0 15px;
        }
        
        .weekly-tasks-table th {
            background-color: #f8f9fa;
            font-weight: bold;
        }
        
        #taskForm {
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
    </style>
</head>
<body class="container mt-4">
    <!-- بیلبورد اطلاعات -->
    <div class="info-billboard mb-4">
        <div class="d-flex flex-wrap justify-content-around">
            <div class="billboard-item">
                <i class="fas fa-tasks me-2"></i>
                <span>تعداد کل وظایف: <?= toPersianNumbers($totalTasks) ?></span>
            </div>
            <div class="billboard-item">
                <i class="fas fa-users me-2"></i>
                <span>تعداد افراد: <?= toPersianNumbers($totalPeople) ?></span>
            </div>
            <div class="billboard-item">
                <i class="fas fa-list-alt me-2"></i>
                <span>انواع وظایف: <?= toPersianNumbers($totalTaskTypes) ?></span>
            </div>
            <div class="billboard-item">
                <i class="fas fa-calendar-day me-2"></i>
                <span>امروز: <?= $todayShamsi ?></span>
            </div>
            <div class="billboard-item">
                <i class="fas fa-clock me-2"></i>
                <span>وظایف امروز: <?= toPersianNumbers($todayTasks) ?></span>
            </div>
        </div>
    </div>

    <!-- بیلبورد متحرک -->
    <div class="marquee-container">
        <div class="marquee-content">
            <?php 
            if(!empty($assignments)) {
                $content = '';
                foreach($assignments as $a) {
                    $taskName = "";
                    foreach($tasks as $t) if($t['task_id'] == $a['task_id']) $taskName = $t['title'];
                    
                    $personName = "";
                    foreach($people as $p) if($p['id'] == $a['person_id']) $personName = $p['name'];
                    
                    $dateShamsi = toJalaliDate($a['date']);
                    $shift = $a['shift'] ?? '-';
                    
                    $content .= '<span class="marquee-item" style="color:'.$a['color'].'">';
                    $content .= '<i class="fas fa-tasks"></i> '.htmlspecialchars("$taskName ($personName - تاریخ: $dateShamsi - شیفت: $shift)");
                    $content .= '</span>';
                }
                echo $content . $content; // نمایش دو باره برای حرکت روان
            } else {
                echo '<span class="marquee-item">هیچ وظیفه‌ای ثبت نشده است</span>';
            }
            ?>
        </div>
    </div>

    <div class="mb-3 d-flex flex-wrap gap-2">
        <button type="button" class="btn btn-primary" onclick="toggleForm(true)">تقسیم کارها</button>
        <button class="btn btn-info" onclick="window.location.href='manage_tasks.php'">افزودن کار جدید</button>
        <button class="btn btn-secondary" onclick="window.location.href='manage_people.php'">📁 مدیریت افراد</button>
        <a href="task_logs.php" class="btn btn-outline-dark">📄 تاریخچه</a>
        <a href="backup.php" class="btn btn-outline-dark">🧾 بکاپ</a>
        <button class="btn btn-warning" onclick="takeScreenshot()">📸 ذخیره تصویر</button>
    </div>

    <div id="taskForm" style="display:<?= isset($autoShowForm) && $autoShowForm ? 'block' : 'none' ?>;">
        <h2><?= $editIndex>=0 ? "ویرایش وظیفه #".($editIndex+1) : "افزودن وظیفه" ?></h2>
        <form method="post" class="row g-3">
            <input type="hidden" name="index" value="<?= $editIndex ?>">
            
            <div class="col-md-6">
                <label for="date" class="form-label">تاریخ</label>
                <input type="date" class="form-control" id="date" name="date" 
                       value="<?= htmlspecialchars($editData['date']) ?>" required>
            </div>
            
            <div class="col-md-6">
                <label for="task_id" class="form-label">کار</label>
                <select class="form-select" id="task_id" name="task_id" required>
                    <option value="">انتخاب کنید</option>
                    <?php foreach($tasks as $task): ?>
                    <option value="<?= $task['task_id'] ?>" <?= $task['task_id']==$editData['task_id'] ? 'selected' : '' ?>>
                        <?= htmlspecialchars($task['title']) ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="col-md-6">
                <label for="person_id" class="form-label">فرد مسئول</label>
                <select class="form-select" id="person_id" name="person_id" required>
                    <option value="">انتخاب کنید</option>
                    <?php foreach($people as $person): ?>
                    <option value="<?= $person['id'] ?>" <?= $person['id']==$editData['person_id'] ? 'selected' : '' ?>>
                        <?= htmlspecialchars($person['name']) ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="col-md-3">
                <label for="color" class="form-label">رنگ</label>
                <select class="form-select" id="color" name="color" required>
                    <option value="black" <?= $editData['color']=='black'?'selected':'' ?>>سیاه</option>
                    <option value="red" <?= $editData['color']=='red'?'selected':'' ?>>قرمز</option>
                    <option value="green" <?= $editData['color']=='green'?'selected':'' ?>>سبز</option>
                    <option value="blue" <?= $editData['color']=='blue'?'selected':'' ?>>آبی</option>
                </select>
            </div>
            
            <div class="col-md-3">
                <label for="shift" class="form-label">شیفت</label>
                <select class="form-select" id="shift" name="shift" required>
                    <option value="">انتخاب کنید</option>
                    <option value="صبح" <?= isset($editData['shift'])&&$editData['shift']=='صبح'?'selected':'' ?>>صبح</option>
                    <option value="عصر" <?= isset($editData['shift'])&&$editData['shift']=='عصر'?'selected':'' ?>>عصر</option>
                </select>
            </div>
            
            <div class="col-12">
                <button type="submit" name="save" class="btn btn-success">ذخیره</button>
                <button type="button" class="btn btn-secondary" onclick="toggleForm(false)">انصراف</button>
            </div>
        </form>
    </div>

    <h2 class="mt-4">وظایف هفتگی</h2>
    <div class="table-responsive">
        <table id="weekly-tasks-table" class="table table-bordered text-center">
            <thead>
                <tr>
                    <?php foreach($weekdays as $day): ?>
                    <th><?= $day ?></th>
                    <?php endforeach; ?>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <?php foreach($weekdays as $day): ?>
                    <td>
                        <?php if(!empty($tasksByWeekday[$day])): ?>
                        <ul class="list-unstyled">
                            <?php foreach($tasksByWeekday[$day] as $task): ?>
                            <li><?= $task ?></li>
                            <?php endforeach; ?>
                        </ul>
                        <?php else: ?>
                        -
                        <?php endif; ?>
                    </td>
                    <?php endforeach; ?>
                </tr>
            </tbody>
        </table>
    </div>

    <h2 class="mt-4">لیست کامل وظایف</h2>
    <div class="table-responsive">
        <table class="table table-striped">
            <thead>
                <tr>
                    <th>#</th>
                    <th>تاریخ</th>
                    <th>کار</th>
                    <th>فرد</th>
                    <th>رنگ</th>
                    <th>شیفت</th>
                    <th>عملیات</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach($assignments as $i => $a): 
                    $taskName = "";
                    foreach($tasks as $t) if($t['task_id'] == $a['task_id']) $taskName = $t['title'];
                    
                    $personName = "";
                    foreach($people as $p) if($p['id'] == $a['person_id']) $personName = $p['name'];
                    
                    $dateShamsi = toJalaliDate($a['date']);
                ?>
                <tr>
                    <td><?= toPersianNumbers($i+1) ?></td>
                    <td><?= $dateShamsi ?></td>
                    <td><?= htmlspecialchars($taskName) ?></td>
                    <td><?= htmlspecialchars($personName) ?></td>
                    <td><span style="color:<?= $a['color'] ?>">■</span></td>
                    <td><?= htmlspecialchars($a['shift'] ?? '-') ?></td>
                    <td>
                        <a href="?edit=<?= $i ?>" class="btn btn-sm btn-warning">ویرایش</a>
                        <a href="?delete=<?= $i ?>" class="btn btn-sm btn-danger" 
                           onclick="return confirm('آیا مطمئن هستید؟')">حذف</a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/html2canvas@1.4.1/dist/html2canvas.min.js"></script>
    <script>
    function toggleForm(show) {
        const form = document.getElementById('taskForm');
        form.style.display = show ? 'block' : 'none';
        
        if(show) {
            form.scrollIntoView({behavior: 'smooth'});
        }
    }
    
    function takeScreenshot() {
        const element = document.getElementById('weekly-tasks-table');
        html2canvas(element).then(canvas => {
            const link = document.createElement('a');
            link.download = 'وظایف-هفتگی.png';
            link.href = canvas.toDataURL();
            link.click();
        });
    }
    
    // تنظیم خودکار نمایش فرم در صورت ویرایش
    <?php if(isset($autoShowForm) && $autoShowForm): ?>
    document.addEventListener('DOMContentLoaded', function() {
        toggleForm(true);
    });
    <?php endif; ?>
    </script>
</body>
</html>