<?php
// تابع تبدیل عدد انگلیسی به فارسی
function toPersianNumbers($input) {
    $englishNumbers = ['0','1','2','3','4','5','6','7','8','9'];
    $persianNumbers = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹'];
    return str_replace($englishNumbers, $persianNumbers, $input);
}

// تابع تبدیل تاریخ میلادی به شمسی (فرمت: YYYY-MM-DD)
function gregorianToJalali($gy, $gm, $gd) {
    $g_d_m = array(0,31,59,90,120,151,181,212,243,273,304,334);
    if($gy > 1600){
        $jy=979;
        $gy-=1600;
    }else{
        $jy=0;
        $gy-=621;
    }
    $gy2 = ($gm > 2)?($gy+1):$gy;
    $days = (365*$gy) + intval(($gy2+3)/4) - intval(($gy2+99)/100) + intval(($gy2+399)/400) - 80 + $gd + $g_d_m[$gm-1];
    $jy += 33*intval($days/12053);
    $days %= 12053;
    $jy += 4*intval($days/1461);
    $days %= 1461;
    if($days > 365){
        $jy += intval(($days-1)/365);
        $days = ($days-1)%365;
    }
    if($days < 186){
        $jm = 1 + intval($days/31);
        $jd = 1 + ($days % 31);
    }else{
        $jm = 7 + intval(($days-186)/30);
        $jd = 1 + (($days-186) % 30);
    }
    return [$jy, $jm, $jd];
}

// تبدیل تاریخ میلادی به شمسی و نمایش اعداد فارسی
function toJalaliDate($gregorianDate){
    $parts = explode('-', $gregorianDate);
    if(count($parts) !== 3) return $gregorianDate;
    list($gy, $gm, $gd) = $parts;
    list($jy, $jm, $jd) = gregorianToJalali((int)$gy, (int)$gm, (int)$gd);
    $jalaliStr = sprintf('%04d/%02d/%02d', $jy, $jm, $jd);
    return toPersianNumbers($jalaliStr);
}

function logTaskChange($entry) {
    $logFile = __DIR__ . "/data/task_log.json";
    $logs = file_exists($logFile) ? json_decode(file_get_contents($logFile), true) : [];
    $entry['datetime'] = date("Y-m-d H:i:s");
    $entry['by'] = 'admin';
    $logs[] = $entry;
    file_put_contents($logFile, json_encode($logs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// فایل‌های داده
$dataDir = __DIR__ . "/data";
if(!file_exists($dataDir)) mkdir($dataDir);
$peopleFile = "$dataDir/people.json";
$tasksFile  = "$dataDir/tasks.json";
$assignFile = "$dataDir/assignments.json";

$people = json_decode(@file_get_contents($peopleFile), true) ?: [];
$tasks = json_decode(@file_get_contents($tasksFile), true) ?: [];
$assignments = json_decode(@file_get_contents($assignFile), true) ?: [];

// حذف وظیفه
if(isset($_GET['delete'])){
    $index = intval($_GET['delete']);
    if(isset($assignments[$index])){
        $deletedItem = $assignments[$index];
        logTaskChange([
            "action" => "delete",
            "task_id" => $deletedItem['task_id'],
            "person_id" => $deletedItem['person_id'],
            "shift" => $deletedItem['shift'] ?? '-'
        ]);

        array_splice($assignments, $index, 1);
        file_put_contents($assignFile, json_encode($assignments, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
    }
    header("Location: index.php");
    exit;
}

// ویرایش
$editIndex = -1;
$editData = ["date"=>"", "task_id"=>"", "person_id"=>"", "color"=>"black", "shift"=>""];
if(isset($_GET['edit'])){
    $editIndex = intval($_GET['edit']);
    if(isset($assignments[$editIndex])){
        $editData = $assignments[$editIndex];
    }
}

// ذخیره وظیفه
if(isset($_POST['save'])){
    $newData = [
        "date" => $_POST['date'],
        "task_id" => intval($_POST['task_id']),
        "person_id" => intval($_POST['person_id']),
        "color" => $_POST['color'],
        "shift" => $_POST['shift']
    ];

    if($_POST['index'] >= 0){
        $assignments[$_POST['index']] = $newData;
        logTaskChange([
            "action" => "edit",
            "task_id" => $newData['task_id'],
            "person_id" => $newData['person_id'],
            "shift" => $newData['shift']
        ]);
    } else {
        $assignments[] = $newData;
        logTaskChange([
            "action" => "add",
            "task_id" => $newData['task_id'],
            "person_id" => $newData['person_id'],
            "shift" => $newData['shift']
        ]);
    }

    file_put_contents($assignFile, json_encode($assignments, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
    header("Location: index.php");
    exit;
}

// آماده‌سازی داده‌ها بر اساس روز هفته
$weekdays = ['شنبه', 'یکشنبه', 'دوشنبه', 'سه‌شنبه', 'چهارشنبه', 'پنجشنبه', 'جمعه'];
$tasksByWeekday = [];
foreach ($weekdays as $day) $tasksByWeekday[$day] = [];

foreach ($assignments as $a) {
    $timestamp = strtotime($a['date']);
    $dayOfWeek = date('w', $timestamp);
    $mapping = [0=>1, 1=>2, 2=>3, 3=>4, 4=>5, 5=>6, 6=>0];
    $farsiDayName = $weekdays[$mapping[$dayOfWeek]];

    $taskName = "";
    foreach($tasks as $t) if($t['task_id']==$a['task_id']) $taskName=$t['title'];

    $personName = "";
    foreach($people as $p) if($p['id']==$a['person_id']) $personName=$p['name'];

    $dateShamsi = toJalaliDate($a['date']);
    $shift = $a['shift'] ?? '-';
    $tasksByWeekday[$farsiDayName][] = "<span style='color:{$a['color']}'>".htmlspecialchars("$dateShamsi - $taskName ($personName - شیفت: $shift)")."</span>";
}

// محاسبه آمار برای بیلبورد
$totalTasks = count($assignments);
$uniquePeople = array_unique(array_column($assignments, 'person_id'));
$totalPeople = count($uniquePeople);
$uniqueTasks = array_unique(array_column($assignments, 'task_id'));
$totalTaskTypes = count($uniqueTasks);
$todayShamsi = toJalaliDate(date('Y-m-d'));

// تعداد وظایف امروز
$todayTasks = 0;
foreach($assignments as $task) {
    if($task['date'] == date('Y-m-d')) {
        $todayTasks++;
    }
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت وظایف - جدول ایام هفته</title>
    <link href="https://cdn.fontcdn.ir/Font/Persian/Vazir/Vazir.css" rel="stylesheet" crossorigin="anonymous" />
    <link rel="icon" type="image/png" href="img/11.ico">
    <script src="https://cdn.jsdelivr.net/npm/html2canvas@1.4.1/dist/html2canvas.min.js"></script>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="styles.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        /* استایل برای بیلبورد متحرک بهبود یافته */
        .marquee-container {
            width: 100%;
            overflow: hidden;
            background: #f0f8ff;
            padding: 15px 0;
            margin-bottom: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            position: relative;
        }
        
        .marquee-content {
            display: inline-block;
            white-space: nowrap;
            padding-right: 100%;
            animation: marquee 60s linear infinite;
        }
        
        .marquee-item {
            display: inline-block;
            margin: 0 40px;
            font-size: 16px;
            font-weight: 500;
            padding: 5px 10px;
            border-radius: 5px;
            background: rgba(255,255,255,0.2);
        }
        
        @keyframes marquee {
            0% { transform: translateX(0); }
            100% { transform: translateX(-50%); }
        }
        
        /* استایل برای بیلبورد اطلاعات */
        .info-billboard-container {
            width: 100%;
            overflow-x: auto;
            margin-bottom: 20px;
            background: #f8f9fa;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .info-billboard {
            display: flex;
            padding: 15px;
            min-width: max-content;
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            color: white;
            border-radius: 8px;
        }
        
        .billboard-item {
            display: flex;
            align-items: center;
            margin: 0 25px;
            font-size: 16px;
            white-space: nowrap;
        }
        
        .billboard-item i {
            margin-left: 8px;
            font-size: 20px;
        }
        
        /* بهبود ظاهر جدول */
        .weekly-tasks-table th {
            background-color: #f8f9fa;
            font-weight: bold;
            text-align: center;
        }
        
        /* استایل‌های عمومی */
        body {
            font-family: Vazir, sans-serif;
        }
        
        .btn {
            font-family: Vazir, sans-serif;
        }
        
        .delete-btn {
            cursor: pointer;
            background: none;
            border: none;
            padding: 0 5px;
            margin: 0;
            color: #dc3545;
            transition: all 0.3s;
        }
        
        .delete-btn:hover {
            color: #a71d2a;
            transform: scale(1.2);
        }
    </style>
</head>
<body class="container mt-4">
    <!-- بیلبورد اطلاعات -->
    <div class="info-billboard-container">
        <div class="info-billboard">
            <div class="billboard-item">
                <i class="fas fa-tasks"></i>
                <span>تعداد کل وظایف: <?= toPersianNumbers($totalTasks) ?></span>
            </div>
            
            <div class="billboard-item">
                <i class="fas fa-users"></i>
                <span>تعداد افراد: <?= toPersianNumbers($totalPeople) ?></span>
            </div>
            
            <div class="billboard-item">
                <i class="fas fa-list-alt"></i>
                <span>انواع وظایف: <?= toPersianNumbers($totalTaskTypes) ?></span>
            </div>
            
            <div class="billboard-item">
                <i class="fas fa-calendar-day"></i>
                <span>امروز: <?= $todayShamsi ?></span>
            </div>
            
            <div class="billboard-item">
                <i class="fas fa-clock"></i>
                <span>وظایف امروز: <?= toPersianNumbers($todayTasks) ?></span>
            </div>
        </div>
    </div>

    <div class="mb-3 d-flex flex-wrap gap-2 align-items-center">
        <button type="button" class="btn btn-primary" onclick="toggleForm()">تقسیم کارها</button>
        <button class="btn btn-info" onclick="window.location.href='manage_tasks.php'">افزودن کار جدید</button>
        <button class="btn btn-secondary" onclick="window.location.href='manage_people.php'">📁 مدیریت افراد</button>
        <a href="task_logs.php" class="btn btn-outline-dark">📄 تاریخچه کارها</a>
        <a href="backup.php" class="btn btn-outline-dark">🧾 ایجاد بکاپ کامل</a>
        <button class="btn btn-warning" onclick="takeScreenshot()">📸 ذخیره تصویر از جدول هفتگی</button>
    </div>

    <div id="taskForm" style="display:none; margin-top:20px;">
        <h2><?= $editIndex>=0 ? "ویرایش وظیفه" : "افزودن وظیفه" ?></h2>
        <form method="post" action="index.php">
            <input type="hidden" name="index" value="<?= $editIndex ?>">
            
            <div class="mb-2">
                <label for="date">تاریخ:</label>
                <input type="date" id="date" name="date" value="<?= htmlspecialchars($editData['date']) ?>" required>
            </div>

            <div class="mb-2">
                <label for="task_id">کار:</label>
                <select id="task_id" name="task_id" required>
                    <option value="">-- انتخاب کار --</option>
                    <?php foreach($tasks as $task): ?>
                        <option value="<?= $task['task_id'] ?>" <?= $task['task_id']==$editData['task_id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($task['title']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="mb-2">
                <label for="color">رنگ:</label>
                <select id="color" name="color" required>
                    <option value="black" <?= ($editData['color'] == 'black') ? 'selected' : '' ?>>سیاه</option>
                    <option value="#FFD700" <?= ($editData['color'] == '#FFD700') ? 'selected' : '' ?> style="background:#FFD700;color:black;">طلایی</option>
                    <option value="red" <?= ($editData['color'] == 'red') ? 'selected' : '' ?>>قرمز</option>
                    <option value="green" <?= ($editData['color'] == 'green') ? 'selected' : '' ?>>سبز</option>
                </select>
            </div>
            
            <div class="mb-2">
                <label for="shift">شیفت:</label>
                <select id="shift" name="shift" required>
                    <option value="">-- انتخاب شیفت --</option>
                    <option value="صبح" <?= (isset($editData['shift']) && $editData['shift'] == 'صبح') ? 'selected' : '' ?>>صبح</option>
                    <option value="عصر" <?= (isset($editData['shift']) && $editData['shift'] == 'عصر') ? 'selected' : '' ?>>عصر</option>
                </select>
            </div>

            <div class="mb-2">
                <label for="person_id">فرد مسئول:</label>
                <select id="person_id" name="person_id" required>
                    <option value="">-- انتخاب فرد --</option>
                    <?php foreach($people as $person): ?>
                        <option value="<?= $person['id'] ?>" <?= $person['id']==$editData['person_id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($person['name']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <button type="submit" name="save" class="btn btn-success">ذخیره</button>
        </form>
    </div>

    <h2 class="mt-4">وظایف بر اساس ایام هفته</h2>
    <table id="weekly-tasks-table" class="table table-bordered text-center weekly-tasks-table">
        <tr>
            <?php foreach($weekdays as $day): ?>
                <th><?= htmlspecialchars($day) ?></th>
            <?php endforeach; ?>
        </tr>
        <tr>
            <?php foreach($weekdays as $day): ?>
                <td>
                    <?php if(count($tasksByWeekday[$day]) > 0): ?>
                        <ul style="list-style-type: none; padding-right: 0; margin: 0;">
                            <?php foreach($tasksByWeekday[$day] as $index => $taskItem): ?>
                                <?php 
                                    // پیدا کردن index واقعی در آرایه assignments
                                    $realIndex = null;
                                    foreach($assignments as $i => $a) {
                                        $taskName = "";
                                        foreach($tasks as $t) if($t['task_id'] == $a['task_id']) $taskName = $t['title'];
                                        $personName = "";
                                        foreach($people as $p) if($p['id'] == $a['person_id']) $personName = $p['name'];
                                        $dateShamsi = toJalaliDate($a['date']);
                                        $shift = $a['shift'] ?? '-';
                                        $currentItem = "<span style='color:{$a['color']}'>".htmlspecialchars("$dateShamsi - $taskName ($personName - شیفت: $shift)")."</span>";
                                        
                                        if($currentItem == $taskItem) {
                                            $realIndex = $i;
                                            break;
                                        }
                                    }
                                ?>
                                <li style="display: flex; align-items: center; gap: 5px; justify-content: flex-start;">
                                    <button onclick="confirmDelete(<?= $realIndex ?>)" class="delete-btn">
                                        <i class="fas fa-trash-alt"></i>
                                    </button>
                                    <?= $taskItem /* حاوی تگ HTML هست */ ?>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    <?php else: ?> 
                        -
                    <?php endif; ?>
                </td>
            <?php endforeach; ?>
        </tr>
    </table>

    <h2 class="mt-4">لیست کارها:</h2>
    <table class="table table-striped text-center">
        <tr><th>#</th><th>تاریخ</th><th>کار</th><th>فرد</th><th>رنگ</th><th>شیفت</th><th>عملیات</th></tr>
        <?php foreach($assignments as $i => $a): 
            $taskName = "";
            foreach($tasks as $t) if($t['task_id'] == $a['task_id']) $taskName = $t['title'];
            $personName = "";
            foreach($people as $p) if($p['id'] == $a['person_id']) $personName = $p['name'];
            $dateShamsi = toJalaliDate($a['date']);
        ?>
        <tr>
            <td><?= toPersianNumbers($i+1) ?></td>
            <td><?= $dateShamsi ?></td>
            <td><?= htmlspecialchars($taskName) ?></td>
            <td><?= htmlspecialchars($personName) ?></td>
            <td><span style="color:<?= $a['color'] ?>;">■</span></td>
            <td><?= htmlspecialchars($a['shift'] ?? '-') ?></td>
            <td>
                <!-- <a href="?edit=<?= $i ?>" class="btn btn-sm btn-warning">ویرایش</a>-->
                <a href="confirm_delete.php?index=<?= $i ?>" class="btn btn-sm btn-danger">حذف</a>
            </td>
        </tr>
        <?php endforeach; ?>
    </table>

    <script>
    // تابع نمایش/مخفی کردن فرم
    function toggleForm() {
        var form = document.getElementById('taskForm');
        form.style.display = (form.style.display === 'none') ? 'block' : 'none';
    }
    
    // تابع ذخیره تصویر از جدول
    function takeScreenshot() {
        const element = document.getElementById('weekly-tasks-table');
        if (!element) {
            alert("عنصر جدول پیدا نشد!");
            return;
        }

        html2canvas(element, {
            backgroundColor: '#fff',
            scale: 2,
            useCORS: true,
        }).then(canvas => {
            const link = document.createElement('a');
            link.download = 'tasks_weekly_screenshot.png';
            link.href = canvas.toDataURL('image/png');
            link.click();
        }).catch(e => {
            alert('خطا در گرفتن اسکرین‌شات: ' + e.message);
        });
    }
    
    // تابع تأیید و حذف وظیفه
  function confirmDelete(index) {
    window.location.href = 'confirm_delete.php?index=' + index;
}

    
    // اسکریپت پیشرفته برای بیلبورد متحرک
    document.addEventListener('DOMContentLoaded', function() {
        const marquee = document.querySelector('.marquee-content');
        if(marquee) {
            // محاسبه عرض واقعی محتوا
            const contentWidth = marquee.scrollWidth / 2;
            
            // تنظیم مدت زمان انیمیشن بر اساس عرض محتوا
            const duration = Math.max(60, contentWidth / 30); // حداقل 60 ثانیه
            
            // اعمال انیمیشن
            marquee.style.animationDuration = duration + 's';
            
            // توقف موقت هنگام هاور
            marquee.addEventListener('mouseenter', function() {
                this.style.animationPlayState = 'paused';
            });
            
            marquee.addEventListener('mouseleave', function() {
                this.style.animationPlayState = 'running';
            });
        }
    });
    </script>
</body>
</html>